'use strict';

/* --------------------------------------------------------------
 slider_responsive.js 2017-05-12
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2017 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Slider Responsive Module
 *
 * This module will handle the image replacement whenever the viewport breakpoint changes.
 */
gambio.widgets.module('slider_responsive', [gambio.source + '/libs/responsive', gambio.source + '/libs/events'], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        defaults = {},
        options = $.extend(true, {}, defaults, data),
        module = {},
        slider;

    // ########## PRIVATE FUNCTIONS ##########

    var _breakpointHandler = function _breakpointHandler(event, currentBreakpoint) {
        var $swiperContainer = $this.find('.swiper-container'),
            $swiperWrapper = $swiperContainer.find('.swiper-wrapper'),
            previousSwiperInstance = $swiperContainer.get(0).swiper;

        // Reset the existing swiper instance (if any).
        if (previousSwiperInstance) {
            try {
                previousSwiperInstance.destroy(true, true);
            } catch (exception) {
                // Sometime the breakpoint handler is called many times from various events which leads
                // to errors while destroying previous Swiper instances, thus the try-catch block.
            }
        }
        $swiperWrapper.empty();
        $this.find('.swiper-pagination').empty();

        // Update the slider HTML markup with the breakpoint-respective image.
        slider.slides.forEach(function (slide) {
            slide.images.forEach(function (image) {
                if ((image.breakpoint === currentBreakpoint.name || image.breakpoint === 'xs' && currentBreakpoint.name === 'too small') && image.languageId === parseInt(jse.core.registry.get('languageId')) && image.image !== '') {

                    var $swiperSlide = $('<div class="swiper-slide"></div>');

                    // Are there image areas?
                    var hasAreas = image.areas && image.areas.length;

                    // Randomly generated string.
                    var imageMapId = Math.random().toString(36).substr(2, 5);

                    if (slide.thumbnail !== '') {
                        $swiperSlide.attr({
                            'data-thumb-image': jse.core.config.get('appUrl') + '/images/slider_images/thumbnails/' + slide.thumbnail,
                            'data-thumb-text': slide.title
                        });
                    }

                    var $slideImage = $('<img />');

                    // Use image map resizer plugin to adjust image map area sizes.
                    $slideImage.rwdImageMaps();

                    // Assign image map, if there are image areas.
                    if (hasAreas) {
                        $slideImage.attr('usemap', '#' + imageMapId);
                    }

                    $slideImage.attr({
                        class: 'img-responsive center-block',
                        src: jse.core.config.get('appUrl') + '/images/slider_images/' + image.image,
                        alt: slide.altText,
                        title: slide.title
                    }).appendTo($swiperSlide);

                    if (slide.url) {
                        $slideImage.wrap('<a />').parent().attr({
                            href: slide.url,
                            target: slide.urlTarget
                        });
                    }

                    // Check for image areas and iterate over them.
                    if (hasAreas) {
                        // Create image map element.
                        var $map = $('<map name="' + imageMapId + '">');

                        /**
                         * Iterator function which processes every image area data.
                         * @param {Object} area Image area data.
                         */
                        var imageAreaIterator = function imageAreaIterator(area) {
                            var areaElementOptions = {
                                shape: 'poly',
                                coords: area.coordinates,
                                href: area.linkUrl,
                                title: area.linkTitle,
                                target: area.linkTarget,
                                'data-id': area.id
                            };

                            // Create image area element.
                            var $area = $('<area>', areaElementOptions);

                            // Put area into image map element.
                            $map.append($area);
                        };

                        // Process every image area.
                        image.areas.forEach(imageAreaIterator);

                        // Append image map to slide element.
                        $swiperSlide.append($map);
                    }

                    $swiperSlide.appendTo($swiperWrapper);
                }
            });
        });

        if ($swiperWrapper.children().length === 0) {
            return; // There is no slide set for this breakpoint.
        }

        $swiperContainer.attr({
            'data-gambio-widget': 'swiper',
            'data-swiper-disable-translucence-fix': 'true'
        });

        $swiperContainer.data('swiper-breakpoints', [{
            breakpoint: 100,
            usePreviewBullets: true,
            slidesPerView: 1
        }]);

        $swiperContainer.data('swiper-slider-options', {
            effect: options.effect,
            speed: options.speed,
            nextButton: '.js-teaser-slider-next',
            prevButton: '.js-teaser-slider-prev',
            autoplay: slider.speed
        });

        // Initialize the new swiper instance and trigger the widget ready event.
        gambio.widgets.init($swiperContainer);
        $('body').trigger(jse.libs.theme.events.SLIDER_RESPONSIVE_READY());

        $('img[usemap]').rwdImageMaps();
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget.
     */
    module.init = function (done) {
        if ($(options.source).length === 0) {
            return; // There is no JSON source for the slider data.
        }

        slider = JSON.parse($(options.source).text());

        $(document).on('JSENGINE_INIT_FINISHED', function () {
            $('body').on(jse.libs.theme.events.BREAKPOINT(), _breakpointHandler);
            _breakpointHandler({}, jse.libs.theme.responsive.breakpoint());
            $('img[usemap]').rwdImageMaps();
        });

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
